<?php
declare(strict_types=1);
require_once __DIR__ . '/../app/bootstrap.php';

$page = $_GET['page'] ?? 'home';

function render(string $view, array $data = []): void {
  extract($data);
  require __DIR__ . '/../views/layout.php';
}

switch ($page) {
  case 'home':
    $q = trim((string)($_GET['q'] ?? ''));
    if ($q !== '') {
      $products = db_all($db, "SELECT p.*, u.name seller_name FROM products p JOIN users u ON u.id=p.seller_id WHERE p.is_active=1 AND (p.title LIKE ? OR p.description LIKE ?) ORDER BY p.id DESC LIMIT 60", ["%$q%","%$q%"]);
    } else {
      $products = db_all($db, "SELECT p.*, u.name seller_name FROM products p JOIN users u ON u.id=p.seller_id WHERE p.is_active=1 ORDER BY p.id DESC LIMIT 60");
    }
    render('home', ['products'=>$products, 'q'=>$q, 'user'=>auth_user(), 'config'=>$config]);
    break;

  case 'product':
    $id = (int)($_GET['id'] ?? 0);
    $product = db_one($db, "SELECT p.*, u.name seller_name FROM products p JOIN users u ON u.id=p.seller_id WHERE p.id=? AND p.is_active=1", [$id]);
    if (!$product) { http_response_code(404); echo "Produto não encontrado."; exit; }
    render('product', ['product'=>$product, 'user'=>auth_user(), 'config'=>$config]);
    break;

  case 'register':
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
      csrf_check();
      $name = trim((string)($_POST['name'] ?? ''));
      $email = trim((string)($_POST['email'] ?? ''));
      $pass = (string)($_POST['password'] ?? '');
      if ($name===''||$email===''||strlen($pass)<6) {
        $error = "Preencha tudo. Senha mínimo 6.";
      } else {
        $error = auth_register($db,$name,$email,$pass);
        if (!$error) {
          auth_login($db,$email,$pass);
          redirect('/public/index.php?page=home');
        }
      }
    }
    render('register', ['error'=>$error ?? null, 'user'=>auth_user(), 'config'=>$config]);
    break;

  case 'login':
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
      csrf_check();
      $email = trim((string)($_POST['email'] ?? ''));
      $pass = (string)($_POST['password'] ?? '');
      $u = auth_login($db,$email,$pass);
      if ($u) redirect('/public/index.php?page=home');
      $error = "Login inválido.";
    }
    render('login', ['error'=>$error ?? null, 'user'=>auth_user(), 'config'=>$config]);
    break;

  case 'logout':
    auth_logout();
    redirect('/public/index.php?page=home');
    break;

  case 'cart':
    render('cart', ['user'=>auth_user(), 'config'=>$config]);
    break;

  case 'checkout':
    $user = auth_require();
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') redirect('/public/index.php?page=cart');
    csrf_check();

    $cartJson = (string)($_POST['cart_json'] ?? '[]');
    $items = json_decode($cartJson, true);
    if (!is_array($items) || count($items)===0) { $error="Carrinho vazio."; render('cart', ['error'=>$error,'user'=>$user,'config'=>$config]); break; }

    $deliveryType = (string)($_POST['delivery_type'] ?? 'delivery');
    $address = trim((string)($_POST['delivery_address'] ?? ''));
    $notes = trim((string)($_POST['delivery_notes'] ?? ''));

    // Recalcula preço no servidor (não confia no JS)
    $total = 0;
    $serverItems = [];
    foreach ($items as $it) {
      $pid = (int)($it['id'] ?? 0);
      $qty = max(1, (int)($it['qty'] ?? 1));
      $p = db_one($db, "SELECT id,seller_id,title,price_cents,stock,is_active FROM products WHERE id=? AND is_active=1", [$pid]);
      if (!$p) continue;
      $qty = min($qty, (int)$p['stock']);
      if ($qty <= 0) continue;
      $total += (int)$p['price_cents'] * $qty;
      $serverItems[] = ['p'=>$p,'qty'=>$qty];
    }

    if ($total <= 0 || count($serverItems)===0) { $error="Não foi possível montar o pedido (estoque/itens inválidos)."; render('cart', ['error'=>$error,'user'=>$user,'config'=>$config]); break; }

    $db->beginTransaction();
    try {
      $orderId = db_exec($db, "INSERT INTO orders (buyer_id,total_cents,status,delivery_type,delivery_address,delivery_notes) VALUES (?,?, 'new', ?, ?, ?)", [
        $user['id'], $total, $deliveryType, $address ?: null, $notes ?: null
      ]);

      foreach ($serverItems as $x) {
        $p = $x['p'];
        $qty = $x['qty'];

        db_exec($db, "INSERT INTO order_items (order_id,product_id,seller_id,qty,price_cents,title_snapshot) VALUES (?,?,?,?,?,?)", [
          $orderId, $p['id'], $p['seller_id'], $qty, $p['price_cents'], $p['title']
        ]);

        // baixa estoque simples
        $st = $db->prepare("UPDATE products SET stock = stock - ? WHERE id=? AND stock >= ?");
        $st->execute([$qty, $p['id'], $qty]);
      }

      $db->commit();
    } catch (Throwable $e) {
      $db->rollBack();
      http_response_code(500);
      echo "Erro ao criar pedido: " . e($e->getMessage());
      exit;
    }

    // Integração com ERP (stub)
    require_once __DIR__ . '/../app/integrations/erp_stub.php';
    erp_notify_new_order($config, $orderId);

    render('checkout_success', ['orderId'=>$orderId, 'user'=>$user, 'config'=>$config]);
    break;

  case 'seller':
    $user = auth_require();
    if (!auth_is_role('seller')) {
      render('seller_become', ['user'=>$user,'config'=>$config]);
      break;
    }
    $products = db_all($db, "SELECT * FROM products WHERE seller_id=? ORDER BY id DESC", [$user['id']]);
    render('seller_dashboard', ['products'=>$products,'user'=>$user,'config'=>$config]);
    break;

  case 'seller_become':
    $user = auth_require();
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
      csrf_check();
      $db->prepare("UPDATE users SET role='seller' WHERE id=?")->execute([$user['id']]);
      $_SESSION['user']['role'] = 'seller';
      redirect('/public/index.php?page=seller');
    }
    render('seller_become', ['user'=>$user,'config'=>$config]);
    break;

  case 'seller_new_product':
    $user = auth_require();
    if (!auth_is_role('seller')) redirect('/public/index.php?page=seller');

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
      csrf_check();
      $title = trim((string)($_POST['title'] ?? ''));
      $desc  = trim((string)($_POST['description'] ?? ''));
      $price = (int)round(((float)str_replace(',','.', (string)($_POST['price'] ?? '0'))) * 100);
      $stock = max(0, (int)($_POST['stock'] ?? 0));
      $city  = trim((string)($_POST['city'] ?? ''));
      if ($title==='' || $price<=0) {
        $error = "Título e preço são obrigatórios.";
      } else {
        db_exec($db, "INSERT INTO products (seller_id,title,description,price_cents,stock,city,is_active) VALUES (?,?,?,?,?,?,1)", [
          $user['id'], $title, $desc ?: null, $price, $stock, $city ?: null
        ]);
        redirect('/public/index.php?page=seller');
      }
    }
    render('seller_new_product', ['error'=>$error ?? null,'user'=>$user,'config'=>$config]);
    break;

  case 'orders':
    $user = auth_require();
    $orders = db_all($db, "SELECT * FROM orders WHERE buyer_id=? ORDER BY id DESC LIMIT 50", [$user['id']]);
    render('orders', ['orders'=>$orders,'user'=>$user,'config'=>$config]);
    break;

  case 'api_ping':
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['ok'=>true,'app'=>$config['APP_NAME'],'time'=>date('c')]);
    break;

  default:
    http_response_code(404);
    echo "Página não encontrada.";
}
