/* Cidade Shop MVP JS
   - carrinho simples via localStorage
*/
const Cart = {
  key: 'cidade_shop_cart_v1',
  read() {
    try { return JSON.parse(localStorage.getItem(this.key) || '[]'); } catch(e){ return []; }
  },
  write(items) { localStorage.setItem(this.key, JSON.stringify(items)); },
  add(item) {
    const items = this.read();
    const idx = items.findIndex(x => x.id === item.id);
    if (idx >= 0) items[idx].qty += item.qty;
    else items.push(item);
    this.write(items);
  },
  remove(id) {
    const items = this.read().filter(x => x.id !== id);
    this.write(items);
  },
  clear() { this.write([]); },
  totalCents() {
    return this.read().reduce((s,x)=> s + (x.price_cents * x.qty), 0);
  }
};

function cartCount() {
  return Cart.read().reduce((s,x)=> s + x.qty, 0);
}

function updateCartBadge() {
  const el = document.querySelector('[data-cart-badge]');
  if (!el) return;
  el.textContent = cartCount();
}

document.addEventListener('DOMContentLoaded', () => {
  updateCartBadge();

  document.querySelectorAll('[data-add-to-cart]').forEach(btn => {
    btn.addEventListener('click', () => {
      const item = {
        id: parseInt(btn.dataset.id, 10),
        title: btn.dataset.title,
        price_cents: parseInt(btn.dataset.priceCents, 10),
        qty: 1
      };
      Cart.add(item);
      updateCartBadge();
      alert('Adicionado ao carrinho!');
    });
  });

  // Render carrinho na página cart
  const cartRoot = document.querySelector('[data-cart-root]');
  if (cartRoot) {
    const items = Cart.read();
    if (items.length === 0) {
      cartRoot.innerHTML = '<p>Seu carrinho está vazio.</p>';
    } else {
      const rows = items.map(x => `
        <div class="cart-row">
          <div>
            <div class="cart-title">${escapeHtml(x.title)}</div>
            <div class="cart-sub">Qtd: ${x.qty}</div>
          </div>
          <div class="cart-right">
            <div class="cart-price">${fmtMoney(x.price_cents)}</div>
            <button class="btn btn-small btn-danger" data-remove="${x.id}">Remover</button>
          </div>
        </div>
      `).join('');
      cartRoot.innerHTML = rows + `
        <div class="cart-total">
          <strong>Total:</strong> <span>${fmtMoney(Cart.totalCents())}</span>
        </div>
      `;
      cartRoot.querySelectorAll('[data-remove]').forEach(b => {
        b.addEventListener('click', () => {
          Cart.remove(parseInt(b.dataset.remove, 10));
          location.reload();
        });
      });
    }

    const hidden = document.querySelector('input[name="cart_json"]');
    if (hidden) hidden.value = JSON.stringify(Cart.read());
  }

  // Ao finalizar pedido, limpa carrinho
  const clearOnSuccess = document.querySelector('[data-clear-cart]');
  if (clearOnSuccess) {
    Cart.clear();
    updateCartBadge();
  }
});

function fmtMoney(cents){
  const v = cents/100;
  return v.toLocaleString('pt-BR',{style:'currency',currency:'BRL'});
}

function escapeHtml(str){
  return (str||'').replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#039;'}[m]));
}
